package vmm.console;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;

class ConsolePane extends JPanel
{
  private boolean terminated;
  private JTextArea textArea;
  private int startPosition;
  private String lastCommand;

  public ConsolePane()
  {
    textArea = new JTextArea(new ConsolePaneDocument());
    textArea.setLineWrap(true);
    textArea.setWrapStyleWord(false);
    textArea.setEditable(false);
    textArea.getCaret().setVisible(false);

    textArea.addKeyListener(new KeyAdapter()
      {
        public void keyPressed(KeyEvent event)
        {
          if (event.getKeyCode() == KeyEvent.VK_ENTER)
          {
            textArea.append("\n");

            try
            {
              String command = textArea.getText(startPosition, textArea.getDocument().getLength() - startPosition - 1);
              fireCommand(command);
            }
            catch (BadLocationException e)
            {
              System.out.println(e.getMessage()); //***
            }

            startPosition = textArea.getCaretPosition();
            ConsolePaneDocument cpd = (ConsolePaneDocument) textArea.getDocument();
            cpd.setStartPosition(startPosition);
          }

          if (event.getKeyCode() == KeyEvent.VK_Q && event.isControlDown())
          {
            textArea.append("\n");
            startPosition = textArea.getCaretPosition();
            ConsolePaneDocument cpd = (ConsolePaneDocument) textArea.getDocument();
            cpd.setStartPosition(startPosition);

            fireCommand(null);
          }
        }
      });

    textArea.addCaretListener(new CaretListener()
      {
        public void caretUpdate(CaretEvent event)
        {
          if (event.getDot() < startPosition)
          {
            int documentLength = textArea.getDocument().getLength();

            if (startPosition < documentLength)
              textArea.getCaret().setDot(startPosition);
            else
              textArea.getCaret().setDot(documentLength);
          }
        }
      });

    setLayout(new BorderLayout());

    JScrollPane scrollPane = new JScrollPane(textArea);
    add(scrollPane, BorderLayout.CENTER);
    scrollPane.setPreferredSize(new Dimension(600, 400));
  }

  private synchronized void fireCommand(String command)
  {
    textArea.setEditable(false);
    textArea.getCaret().setVisible(false);
    lastCommand = command;
    notifyAll();
  }

  public synchronized void close()
  {
    terminated = true;
    notifyAll();
  }

  public synchronized void write(final String line)
  {
    try
    {
      SwingUtilities.invokeAndWait(new Runnable()
        {
          public void run()
          {
            textArea.append(line);
            startPosition = textArea.getCaretPosition();
            ConsolePaneDocument cpd = (ConsolePaneDocument) textArea.getDocument();
            cpd.setStartPosition(startPosition);
          }
        });
      }
    catch (Exception e)
    {
      System.out.println(e.getMessage()); //***
    }
  }

  public void writeln(String line)
  {
    write(line + "\n");
  }

  public void writeln()
  {
    write("\n");
  }

  public synchronized String readln()
  {
    while (true)
    {
      try
      {
        if (terminated)
          return null;

        SwingUtilities.invokeAndWait(new Runnable()
          {
            public void run()
            {
              textArea.setEditable(true);
              textArea.getCaret().setVisible(true);
            }
          });

        if (lastCommand == null)
          wait();

        String command = lastCommand;
        lastCommand = null;
        return command;
      }
      catch (Exception e)
      {
        System.out.println(e.getMessage()); //***
      }
    }
  }
}
